<?php
// GeoIP Handler for SimplePay - Enhanced with multiple services and caching

class GeoIPHandler {
    private $cacheFile = 'cache/geoip_cache.json';
    private $cacheExpiry = 3600; // 1 hour
    private $services = [
        'ipapi' => 'http://ip-api.com/json/',
        'ipinfo' => 'https://ipinfo.io/',
        'ipgeolocation' => 'https://api.ipgeolocation.io/ipgeo?apiKey=free&ip=',
        'freeipapi' => 'https://freeipapi.com/api/json/'
    ];
    
    public function __construct() {
        if (!file_exists('cache')) {
            mkdir('cache', 0755, true);
        }
        if (!file_exists('logs')) {
            mkdir('logs', 0755, true);
        }
    }
    
    public function getCountryByIP($ip) {
        // Check cache first
        $cached = $this->getCachedResult($ip);
        if ($cached) {
            $this->logGeoIP("Cache hit for IP: {$ip} -> {$cached['country_code']}");
            return $cached;
        }
        
        // Try each service until we get a result
        foreach ($this->services as $serviceName => $baseUrl) {
            try {
                $result = $this->queryService($serviceName, $baseUrl, $ip);
                if ($result && !empty($result['country_code'])) {
                    $this->cacheResult($ip, $result);
                    $this->logGeoIP("Service {$serviceName} success for IP: {$ip} -> {$result['country_code']}");
                    return $result;
                }
            } catch (Exception $e) {
                $this->logGeoIP("Service {$serviceName} failed for IP: {$ip} - " . $e->getMessage(), 'error');
                continue;
            }
        }
        
        // Fallback result
        $fallback = [
            'ip' => $ip,
            'country_code' => 'UNKNOWN',
            'country_name' => 'Unknown',
            'city' => 'Unknown',
            'region' => 'Unknown',
            'isp' => 'Unknown',
            'service' => 'fallback',
            'timestamp' => time()
        ];
        
        $this->logGeoIP("All services failed for IP: {$ip} - using fallback", 'warning');
        return $fallback;
    }
    
    private function queryService($serviceName, $baseUrl, $ip) {
        $url = $baseUrl . $ip;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_USERAGENT, 'SimplePay-GeoIP/1.0');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || !$response) {
            throw new Exception("HTTP {$httpCode} or empty response");
        }
        
        $data = json_decode($response, true);
        if (!$data) {
            throw new Exception("Invalid JSON response");
        }
        
        return $this->normalizeResponse($serviceName, $data, $ip);
    }
    
    private function normalizeResponse($serviceName, $data, $ip) {
        $normalized = [
            'ip' => $ip,
            'service' => $serviceName,
            'timestamp' => time()
        ];
        
        switch ($serviceName) {
            case 'ipapi':
                if (isset($data['status']) && $data['status'] === 'fail') {
                    throw new Exception($data['message'] ?? 'API error');
                }
                $normalized['country_code'] = $data['countryCode'] ?? 'UNKNOWN';
                $normalized['country_name'] = $data['country'] ?? 'Unknown';
                $normalized['city'] = $data['city'] ?? 'Unknown';
                $normalized['region'] = $data['regionName'] ?? 'Unknown';
                $normalized['isp'] = $data['isp'] ?? 'Unknown';
                break;
                
            case 'ipinfo':
                $normalized['country_code'] = $data['country'] ?? 'UNKNOWN';
                $normalized['country_name'] = $this->getCountryName($data['country'] ?? '');
                $normalized['city'] = $data['city'] ?? 'Unknown';
                $normalized['region'] = $data['region'] ?? 'Unknown';
                $normalized['isp'] = $data['org'] ?? 'Unknown';
                break;
                
            case 'ipgeolocation':
                $normalized['country_code'] = $data['country_code2'] ?? 'UNKNOWN';
                $normalized['country_name'] = $data['country_name'] ?? 'Unknown';
                $normalized['city'] = $data['city'] ?? 'Unknown';
                $normalized['region'] = $data['state_prov'] ?? 'Unknown';
                $normalized['isp'] = $data['isp'] ?? 'Unknown';
                break;
                
            case 'freeipapi':
                $normalized['country_code'] = $data['countryCode'] ?? 'UNKNOWN';
                $normalized['country_name'] = $data['countryName'] ?? 'Unknown';
                $normalized['city'] = $data['cityName'] ?? 'Unknown';
                $normalized['region'] = $data['regionName'] ?? 'Unknown';
                $normalized['isp'] = 'Unknown';
                break;
                
            default:
                throw new Exception("Unknown service: {$serviceName}");
        }
        
        return $normalized;
    }
    
    private function getCachedResult($ip) {
        if (!file_exists($this->cacheFile)) {
            return null;
        }
        
        $cache = json_decode(file_get_contents($this->cacheFile), true);
        if (!$cache || !isset($cache[$ip])) {
            return null;
        }
        
        $cached = $cache[$ip];
        if (time() - $cached['timestamp'] > $this->cacheExpiry) {
            return null; // Expired
        }
        
        return $cached;
    }
    
    private function cacheResult($ip, $result) {
        $cache = [];
        if (file_exists($this->cacheFile)) {
            $cache = json_decode(file_get_contents($this->cacheFile), true) ?: [];
        }
        
        $cache[$ip] = $result;
        
        // Keep only last 1000 entries
        if (count($cache) > 1000) {
            $cache = array_slice($cache, -1000, null, true);
        }
        
        file_put_contents($this->cacheFile, json_encode($cache, JSON_PRETTY_PRINT));
    }
    
    private function getCountryName($countryCode) {
        $countries = [
            'JO' => 'Jordan',
            'HU' => 'Hungary',
            'US' => 'United States',
            'GB' => 'United Kingdom',
            'DE' => 'Germany',
            'FR' => 'France',
            'CA' => 'Canada',
            'AU' => 'Australia',
            'JP' => 'Japan',
            'CN' => 'China',
            'IN' => 'India',
            'BR' => 'Brazil',
            'RU' => 'Russia',
            'IT' => 'Italy',
            'ES' => 'Spain',
            'NL' => 'Netherlands',
            'SE' => 'Sweden',
            'NO' => 'Norway',
            'DK' => 'Denmark',
            'FI' => 'Finland'
        ];
        
        return $countries[$countryCode] ?? 'Unknown';
    }
    
    private function logGeoIP($message, $level = 'info') {
        $logFile = 'logs/geoip.log';
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] [{$level}] {$message}\n";
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
}
?>
